import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'

export async function GET(
  request: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    const car = await db.car.findUnique({
      where: { id: params.id }
    })

    if (!car) {
      return NextResponse.json(
        { success: false, error: 'Car not found' },
        { status: 404 }
      )
    }

    return NextResponse.json({
      success: true,
      data: car
    })
  } catch (error) {
    console.error('Get car error:', error)
    return NextResponse.json(
      { success: false, error: 'Internal server error' },
      { status: 500 }
    )
  }
}

export async function PUT(
  request: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    const body = await request.json()
    const {
      name,
      brand,
      model,
      year,
      price,
      image,
      description,
      transmission,
      fuelType,
      seats,
      available
    } = body

    const car = await db.car.update({
      where: { id: params.id },
      data: {
        name,
        brand,
        model,
        year: parseInt(year),
        price: parseFloat(price),
        image,
        description,
        transmission,
        fuelType,
        seats: parseInt(seats),
        available
      }
    })

    return NextResponse.json({
      success: true,
      data: car
    })
  } catch (error) {
    console.error('Update car error:', error)
    return NextResponse.json(
      { success: false, error: 'Internal server error' },
      { status: 500 }
    )
  }
}

export async function DELETE(
  request: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    await db.car.delete({
      where: { id: params.id }
    })

    return NextResponse.json({
      success: true,
      message: 'Car deleted successfully'
    })
  } catch (error) {
    console.error('Delete car error:', error)
    return NextResponse.json(
      { success: false, error: 'Internal server error' },
      { status: 500 }
    )
  }
}